/**
 * \file: mspin_connection_tcp.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN Connection Adapter TCP/IP
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
* \copyright: (c) 2003 - 2016 ADIT Corporation
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_connection_tcp.h"
#include "mspin_connection_tcp_manager.h"
#include "mspin_logging.h"

#include <pthread.h>        //pthread_*
#include <string.h>         //memset
#include <errno.h>          //errno
#include <sys/prctl.h>      //prctl
#include <sys/socket.h>
#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
#include <mqueue.h>         //mq_open,...
#endif //MSPIN_TCP_USE_MESSAGE_QUEUE

#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
#define MSPIN_TCP_READ_SELECT_TIMEOUT_MS 500

static void* mspin_tcp_readThread(void* exinf)
{
    U8 readbuf[MSPIN_TCP_READ_BUFFER_SIZE] = {0};
    S32 rc = 0;
    S32 sendResult = 0;
    fd_set fds;
    struct timeval tv;
    int selectResult = 0;
    int tcpSocket = -1;

    //Set thread name
    rc = prctl(PR_SET_NAME, "mspin_tcpReader", 0, 0, 0);

    mspin_tcpConnectionHandle_t *pTCPHandle = (mspin_tcpConnectionHandle_t*)exinf;

    if (!pTCPHandle)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(exinf=%p) FATAL ERROR: TCP handle is NULL", __FUNCTION__, exinf);

        pthread_exit((void*)exinf);
        return NULL;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) started", __FUNCTION__, exinf);

    //Set cancelability state and type
    rc = pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
    if (rc != 0)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) pthread_setcancelstate() failed %d -> terminate",
                __FUNCTION__, exinf, rc);

        pthread_exit((void*)exinf);
        return NULL;
    }

    //Reader loop
    while (!pTCPHandle->quitReader && (rc >= 0) && (pTCPHandle->socketFD > -1))
    {
        tcpSocket = pTCPHandle->socketFD; //store in local variable which is valid till this iteration

        FD_ZERO(&fds);
        FD_SET(tcpSocket, &fds);
        tv.tv_sec = MSPIN_TCP_READ_SELECT_TIMEOUT_MS/1000; //seconds
        tv.tv_usec = (MSPIN_TCP_READ_SELECT_TIMEOUT_MS%1000)*1000; //milliseconds

        mspin_log_printLn(eMspinVerbosityVerbose, "%s(exinf=%p) wait for data", __FUNCTION__, exinf);

        selectResult = select(tcpSocket+1, &fds, NULL, NULL, &tv);
        if ((selectResult > 0) && FD_ISSET(tcpSocket, &fds))
        {
            //Read data
            mspin_log_printLn(eMspinVerbosityVerbose, "%s(exinf=%p) call read", __FUNCTION__, exinf);

            rc = recv(tcpSocket, (char*)readbuf, MSPIN_TCP_READ_BUFFER_SIZE, 0); //no flags

            mspin_log_printLn(eMspinVerbosityVerbose, "%s(exinf=%p) read returned with rc=%d", __FUNCTION__, exinf, rc);

            if (rc > 0)
            {
                //ToDo: Can it happen that we can't write all bytes (for example when the message is bigger than the messages of the queue)?
                // see error EMSGSIZE of man mq_send
                // => fix also in iAP2 code!!!
                sendResult = mq_send(pTCPHandle->mesgQueue, (char*)readbuf, rc, 0);
                if (sendResult >= 0) //0 is success
                {
                    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) %d bytes received and queued",
                            __FUNCTION__, exinf, rc); //ToDo: change back to verbose level
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(exinf=%p) ERROR: Failed to write data with len=%d into queue with rc=%d -> terminate thread",
                            __FUNCTION__, exinf, rc, sendResult);

                    rc = sendResult;
                }
            }
            else if (0 == rc)
            {
                //When recv returns 0 on a stream socket, it means an ordertly shutdown
                mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: socket shutdown", __FUNCTION__, exinf);

                //Set error to true
                pTCPHandle->error = TRUE;

                //And place a 0 byte packet into queue to let any receive unblock
                (void)mq_send(pTCPHandle->mesgQueue, (char*)readbuf, 0, 0); //do not care for the result

                rc = -1;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(exinf=%p) ERROR: Failed to read data with rc=%d (errno: '%s' (%d)) -> terminate thread",
                        __FUNCTION__, exinf, rc, strerror(errno), errno);
                //leave because rc is negative
            }
        }
        else if (selectResult < 0)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: select() returned with error code=%d and errno='%s'(%d) => abort",
                    __FUNCTION__, selectResult, strerror(errno), errno);
            rc = -1;
        }
        else if (selectResult == 0)
        {
            //Timeout, nothing to do
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: select() indicated data on invalid fd",
                    __FUNCTION__, selectResult);
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(exinf=%p) terminate", __FUNCTION__, exinf);

    pthread_exit((void*)exinf);
    return NULL;
}

//ToDo: create one function for mspin_tcp_startTCPReaderThread and mspin_tcp_startUDPListenerThread
// required: threadID, function pointer, quit flag
static S32 mspin_tcp_startTCPReaderThread(mspin_tcpConnectionHandle_t* context)
{
    pthread_attr_t attr;
    S32 rc = -1;

    memset(&attr, 0, sizeof(pthread_attr_t));

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to init thread attributes", __FUNCTION__);
        return rc;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to set detach state", __FUNCTION__);
        return rc;
    }

    //context->quitThread = FALSE;

    rc = pthread_create(&context->threadID, &attr, mspin_tcp_readThread, (void*)context);
    if (0 == rc)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() using thread id=%d", __FUNCTION__, context->threadID);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to create thread", __FUNCTION__);
        context->threadID = 0;
    }

    (void)pthread_attr_destroy(&attr);

    return rc;
}
#endif //MSPIN_TCP_USE_MESSAGE_QUEUE

S32 mspin_tcp_receive(mspin_connectionHandle_t *pConnectionHandle, U8* buffer, U32 bufferLen, U32 timeout)
{
    S32 rc = -1;

    if (pConnectionHandle && (pConnectionHandle->connectionID > -1))
    {
#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
        //Read from message queue
        if (0 < pTCPContext->mesgQueue && !pTCPContext->error)
        {
            struct timespec currentTime;
            struct timespec absTime;

            mspin_log_printLn(eMspinVerbosityVerbose, "%s(ctx=%p, buffer=%p, len=%d, timeout=%d) wait for data",
                    __FUNCTION__, pTCPContext, buffer, bufferLen, timeout);

            //It would be better to use CLOCK_MONOTONIC for timers in order to avoid
            // problems when the system time is changed. But CLOCK_MONOTONIC can't be used for mq_timedreceive. See
            // http://stackoverflow.com/questions/33873374/pthreads-mq-timedreceive-pthread-cond-timedwait-and-clock-realtime
            // => but this would be only a problem if the time is set back and no data is put into the message queue.

            clock_gettime(CLOCK_REALTIME, &currentTime);

            absTime.tv_sec = currentTime.tv_sec + (timeout / 1000);
            absTime.tv_nsec = currentTime.tv_nsec + ((timeout % 1000)*1000000);

            //Check if nano seconds are bigger than 1000000000ns (=1s) (compare man page of mq_timedreceive)
            if (absTime.tv_nsec >= 1000000000)
            {
                absTime.tv_sec += 1;
                absTime.tv_nsec -= 1000000000;
            }

            rc = (S32)mq_timedreceive(pTCPContext->mesgQueue, (char*)buffer, bufferLen, NULL, &absTime);
            if (rc > 0)
            {
                mspin_log_printLn(eMspinVerbosityVerbose, "%s(ctx=%p, buffer=%p, len=%d, timeout=%d) %d data read",
                        __FUNCTION__, pTCPContext, buffer, bufferLen, timeout, rc);
            }
            else if (0 == rc)
            {
                //0 bytes read: This might be used for unblocking this function
                mspin_log_printLn(eMspinVerbosityInfo, "%s(ctx=%p, buffer=%p, len=%d, timeout=%d) %d data read",
                        __FUNCTION__, pTCPContext, buffer, bufferLen, timeout, rc);
            }
            else // (-1 == rc)
            {
                if (errno == ETIMEDOUT)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(buffer=%p, len=%d) timeout occurred after %dms => return 0",
                            __FUNCTION__, buffer, bufferLen, timeout);
                    rc = 0; //update rc
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(buffer=%p, len=%d) ERROR: Reading data failed with rc=%d (errno='%s')",
                            __FUNCTION__, buffer, bufferLen, rc, strerror(errno));
                }
                //return return code of mq_receive
            }
        }
        else if (pTCPContext->error)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(buffer=%p, len=%d) ERROR: Error set => return error",
                    __FUNCTION__, buffer, bufferLen);
            //return default error value
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(buffer=%p, len=%d) ERROR: No open read message queue",
                    __FUNCTION__, buffer, bufferLen);
            //return default error value
        }
#else
        //U8 readbuf[MSPIN_TCP_READ_BUFFER_SIZE] = {0};
        fd_set fds = {{0}};
        struct timeval tv;
        int selectResult = 0;
        int tcpSocket = -1;

        tcpSocket = pConnectionHandle->connectionID; //store in local variable which is valid till this iteration
        /* PRQA: Lint Message 529: Warning due symbol __d0 not subsequently referenced in the asm section. Not an issue */
        /*lint -save -e529*/
        FD_ZERO(&fds);
        /*lint -restore*/
        FD_SET(tcpSocket, &fds);
        tv.tv_sec = timeout/1000; //seconds
        tv.tv_usec = (timeout%1000)*1000; //milliseconds

        mspin_log_printLn(eMspinVerbosityVerbose, "%s(buffer=%p, len=%d) wait for data",
                __FUNCTION__, buffer, bufferLen);

        selectResult = select(tcpSocket+1, &fds, NULL, NULL, &tv);
        if ((selectResult > 0) && FD_ISSET(tcpSocket, &fds))
        {
            //Read data
            mspin_log_printLn(eMspinVerbosityVerbose, "%s(buffer=%p, len=%d) call read",
                    __FUNCTION__, buffer, bufferLen);

            rc = recv(tcpSocket, (char*)buffer, bufferLen, 0); //no flags

            mspin_log_printLn(eMspinVerbosityVerbose, "%s(buffer=%p, len=%d) read returned with rc=%d",
                    __FUNCTION__, buffer, bufferLen, rc);

            if (rc > 0)
            {
                mspin_log_printLn(eMspinVerbosityVerbose, "%s(buffer=%p, len=%d) %d bytes read",
                        __FUNCTION__, buffer, bufferLen, rc);
            }
            else if (0 == rc)
            {
                //When recv returns 0 on a stream socket, it means an orderly shutdown
                mspin_log_printLn(eMspinVerbosityError, "%s(buffer=%p, len=%d) ERROR: socket shutdown",
                        __FUNCTION__, buffer, bufferLen);

                //Set error to true
                (void)mspin_tcp_setError(tcpSocket, TRUE);

                rc = -1;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(buffer=%p, len=%d) ERROR: Failed to read data on socket=%d with rc=%d (errno: '%s' (%d)) -> terminate thread",
                        __FUNCTION__, buffer, bufferLen, tcpSocket, rc, strerror(errno), errno);
                //return negative rc
            }
        }
        else if (selectResult < 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: select() returned with error code=%d and errno='%s'(%d) => abort",
                    __FUNCTION__, selectResult, strerror(errno), errno);
            rc = -1;
        }
        else if (selectResult == 0)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(buffer=%p, len=%d) timeout",
                    __FUNCTION__, buffer, bufferLen, rc);
            rc = 0;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: select() indicated data on invalid fd",
                    __FUNCTION__, selectResult);
        }
#endif // MSPIN_TCP_USE_MESSAGE_QUEUE
    }
    else if (!pConnectionHandle)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Connection handle is NULL",
                __FUNCTION__, pConnectionHandle, buffer, bufferLen);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Connection ID =%d is invalid",
                __FUNCTION__, pConnectionHandle, buffer, bufferLen, pConnectionHandle->connectionID);
    }

    return rc;
}

S32 mspin_tcp_send(mspin_connectionHandle_t *pConnectionHandle, const U8* buffer, U32 bufferLen)
{
    S32 rc = 0;
    U32 transferredUntilNow = 0;
    int socketFD = -1;

    if (pConnectionHandle)
    {
        socketFD = pConnectionHandle->connectionID;

        while ((bufferLen > transferredUntilNow) && (rc >= 0))
        {
            if ((socketFD > -1) && (!mspin_tcp_getError(socketFD)))
            {
                rc = send(socketFD, buffer+transferredUntilNow, bufferLen-transferredUntilNow, 0);
                if (rc > 0)
                {
                    transferredUntilNow += rc;
                    if (transferredUntilNow == bufferLen)
                    {
                        mspin_log_printLn(eMspinVerbosityDebug, "%s(connHandle=%p, buffer=%p, bufferLen=%d) all %d bytes sent",
                                __FUNCTION__, pConnectionHandle, buffer, bufferLen, transferredUntilNow);
                    }
                    else if (transferredUntilNow < bufferLen)
                    {
                        mspin_log_printLn(eMspinVerbosityDebug, "%s(connHandle=%p, buffer=%p, bufferLen=%d) %d/%d bytes sent -> try sending remaining bytes",
                                __FUNCTION__, pConnectionHandle, buffer, bufferLen, transferredUntilNow, bufferLen);
                    }
                    else
                    {
                        mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Too many bytes=%d sent",
                                __FUNCTION__, pConnectionHandle, buffer, bufferLen, rc);
                    }
                }
                else if (rc == 0)
                {
                    mspin_log_printLn(eMspinVerbosityWarn, "%s(connHandle=%p, buffer=%p, bufferLen=%d) WARNING: 0 bytes sent => try again",
                            __FUNCTION__, pConnectionHandle, buffer, bufferLen);
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Sending bytes failed with '%s'(%d)",
                            __FUNCTION__, pConnectionHandle, buffer, bufferLen, strerror(errno), errno);

                    //Set error to true
                    (void)mspin_tcp_setError(socketFD, TRUE);
                }
            }
            else if (mspin_tcp_getError(socketFD))
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Error set => do not sent (socketFD=%d)",
                        __FUNCTION__, pConnectionHandle, buffer, bufferLen, socketFD);
                rc = -1;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Invalid socket (socketFD=%d)",
                        __FUNCTION__, pConnectionHandle, buffer, bufferLen, socketFD);
                rc = -1;
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Connection handle is NULL",
                __FUNCTION__, pConnectionHandle, buffer, bufferLen);
        rc = -1;
    }

    return rc;
}
